﻿using System;
using System.Linq;
using GTA;
using GTA.Math;
using GTA.Native;

namespace udaloyFunnels
{
    public sealed class udaloyFunnels : Script
    {
        private LoopedPTFX[] funnels;

        private Vector3[] offsets =
        {
			new Vector3(-3, 0.0f, 16.0f),
			new Vector3(3, 0.0f, 16.0f),
            new Vector3(-3, -19.5f, 16.0f),
			new Vector3(3, -19.5f, 16.0f),
        };

        private bool bVehicledInit = false;

        private readonly Ped PlayerPed =
            Game.Player.Character;

        public udaloyFunnels()
        {
            funnels = Enumerable.Range(0, 4)
                .Select(x => new LoopedPTFX("core", "ent_amb_smoke_foundry_white"))
                .ToArray();

            Tick += RunFrame;
        }

        private void RunFrame(object sender, EventArgs e)
        {
            if (funnels.Length < 0) return;

            if (PlayerPed.IsInVehicle() && (uint)PlayerPed.CurrentVehicle.Model.Hash == 0x5B11AD62 &&
                PlayerPed.CurrentVehicle.IsDriveable)
            {
                if (!bVehicledInit)
                {
                    for (int x = 0; x < 4; x++)
                    {
                        funnels[x].Start(PlayerPed.CurrentVehicle, offsets[x], 1.5f);
                    }

                    bVehicledInit = true;
                }

                float scale = 1.5f + PlayerPed.CurrentVehicle.Speed / 100;

                if (scale >= 2.0f) scale = 2.0f;

                for (int i = 0; i < 4; i++)
                {
                    funnels[i].Scale = scale;

                    funnels[i].SetOffsets(offsets[i] + new Vector3(0, 0, PlayerPed.CurrentVehicle.Speed * 0.6f), PlayerPed.CurrentVehicle.Rotation);
                }
            }

            else RemoveFunnels();
        }

        private void RemoveFunnels()
        {
            if (bVehicledInit)
            {
                for (int i = 0; i < 4; i++)
                {
                    funnels[i].Remove();
                }

                bVehicledInit = false;
            }
        }

        protected override void Dispose(bool A_0)
        {
            RemoveFunnels();
            base.Dispose(A_0);
        }
    }

    public class LoopedPTFX
    {
        private float scale;

        public int Handle { get; private set; }
        public string AssetName { get; private set; }
        public string FXName { get; private set; }

        /// <summary>
        /// If the particle FX is spawned.
        /// </summary>
        public bool Exists { get { if (Handle == -1) return false; return Function.Call<bool>(Hash.DOES_PARTICLE_FX_LOOPED_EXIST, Handle); } }

        /// <summary>
        /// If the particle FX asset is loaded.
        /// </summary>
        public bool IsLoaded { get { return Function.Call<bool>(Hash.HAS_NAMED_PTFX_ASSET_LOADED, AssetName); } }

        /// <summary>
        /// Set the particle FX scale.
        /// </summary>
        public float Scale { get { return scale; } set { Function.Call(Hash.SET_PARTICLE_FX_LOOPED_SCALE, Handle, scale = value); } }

        ///// <summary>
        ///// Set the particle FX looped colour.
        ///// </summary>
        //public Color Colour { set { Function.Call(Hash.SET_PARTICLE_FX_LOOPED_COLOUR, Handle, value.R, value.G, value.B, 0); } }

        public LoopedPTFX(string assetName, string fxName)
        {
            Handle = -1;
            AssetName = assetName;
            FXName = fxName;
            Load();
        }

        /// <summary>
        /// Load the particle FX asset.
        /// </summary>
        public void Load()
        {
            if (!IsLoaded)
            {
                Function.Call(Hash.REQUEST_NAMED_PTFX_ASSET, AssetName);
            }
        }

        /// <summary>
        /// Start particle FX on the specified entity.
        /// </summary>
        /// <param name="entity">Entity to attach to.</param>
        /// <param name="scale">Scale of the fx.</param>
        /// <param name="offset">Optional offset.</param>
        /// <param name="rotation">Optional rotation.</param>
        /// <param name="bone">Entity bone.</param>
        public void Start(Entity entity, Vector3 offset, float scale, Vector3 rotation, Bone? bone)
        {
            if (Handle != -1) return;

            this.scale = scale;

            Function.Call(Hash._SET_PTFX_ASSET_NEXT_CALL, AssetName);

            Handle = bone == null ?
                Function.Call<int>(Hash.START_PARTICLE_FX_LOOPED_ON_ENTITY, FXName,
                entity, offset.X, offset.Y, offset.Z, rotation.X, rotation.Y, rotation.Z, scale, 0, 0, 1) :
                Function.Call<int>(Hash._START_PARTICLE_FX_LOOPED_ON_ENTITY_BONE, FXName,
                entity, offset.X, offset.Y, offset.Z, rotation.X, rotation.Y, rotation.Z, (int)bone, scale, 0, 0, 0);
        }

        /// <summary>
        /// Start particle FX on the specified entity.
        /// </summary>
        /// <param name="entity">Entity to attach to.</param>
        /// <param name="scale">Scale of the fx.</param>
        /// <param name="offset">Optional offset.</param>
        public void Start(Entity entity, Vector3 offset, float scale)
        {
            Start(entity, offset, scale, Vector3.Zero, null);
        }

        /// <summary>
        /// Start particle FX on the specified entity.
        /// </summary>
        /// <param name="entity">Entity to attach to.</param>
        /// <param name="scale">Scale of the fx.</param>
        public void Start(Entity entity, float scale)
        {
            Start(entity, Vector3.Zero, scale, Vector3.Zero, null);
        }

        /// <summary>
        /// Start particle FX at the specified position.
        /// </summary>
        /// <param name="position">Position in world space.</param>
        /// <param name="scale">Scale of the fx.</param>
        /// <param name="rotation">Optional rotation.</param>
        public void Start(Vector3 position, float scale, Vector3 rotation)
        {
            if (Handle != -1) return;

            this.scale = scale;

            Function.Call(Hash._SET_PTFX_ASSET_NEXT_CALL, AssetName);

            Handle = Function.Call<int>(Hash.START_PARTICLE_FX_LOOPED_AT_COORD, FXName,
             position.X, position.Y, position.Z, rotation.X, rotation.Y, rotation.Z, scale, 0, 0, 0, 0);
        }

        /// <summary>
        /// Start particle FX at the specified position.
        /// </summary>
        /// <param name="position">Position in world space.</param>
        /// <param name="scale">Scale of the fx.</param>
        public void Start(Vector3 position, float scale)
        {
            Start(position, scale, Vector3.Zero);
        }

        /// <summary>
        /// Set position offsets.
        /// </summary>
        /// <param name="offset"></param>
        /// <param name="rotOffset"></param>
        public void SetOffsets(Vector3 offset, Vector3 rotOffset)
        {
            Function.Call(Hash.SET_PARTICLE_FX_LOOPED_OFFSETS, Handle, offset.X, offset.Y, offset.Z, rotOffset.X, rotOffset.Y, rotOffset.Z);
        }

        /// <summary>
        /// Set custom PTFX evolution variables.
        /// </summary>
        /// <param name="variableName"></param>
        /// <param name="value"></param>
        public void SetEvolution(string variableName, float value)
        {
            Function.Call(Hash.SET_PARTICLE_FX_LOOPED_EVOLUTION, Handle, variableName, value, 0);
        }

        /// <summary>
        /// Remove the particle FX.
        /// </summary>
        public void Remove()
        {
            if (Handle == -1) return;

            Function.Call(Hash.STOP_PARTICLE_FX_LOOPED, Handle, 0);

            Function.Call(Hash.REMOVE_PARTICLE_FX, Handle, 0);
            Handle = -1;
        }

        /// <summary>
        /// Remove the particle FX in range.
        /// </summary>
        public static void Remove(Vector3 position, float radius)
        {
            Function.Call(Hash.REMOVE_PARTICLE_FX_IN_RANGE, position.X, position.Y, position.Z, radius);
        }

        /// <summary>
        /// Unload the loaded particle FX asset.
        /// </summary>
        public void Unload()
        {
            if (IsLoaded)
                Function.Call((Hash)0x5F61EBBE1A00F96D, AssetName);
        }
    }
}
